#!/usr/bin/env bash
#
# build_pad.sh – Convert raw symbols → binary keystream → XOR encrypt
#
# Usage:
#   ./build_pad.sh <line_file> <plaintext_file> <output_cipher_bin>
#
# The script accepts many spellings for cards:
#   * Unicode glyphs (K♠, 10♥, …)
#   * Two‑letter abbreviations (KS, TC, AD, …)
#   * Lower‑case versions (ks, tc, ad, …)
#   * Alternate “T10C” style for ten (optional)
#
# All accepted tokens must appear somewhere in scripts/utils/card_map.txt.
# The mapping file can contain as many aliases as you like – the script will
# load them all and match after normalising the token.

set -euo pipefail

# ----------------------------------------------------------------------
# Helper: print usage and exit
# ----------------------------------------------------------------------
usage() {
    echo "Usage: $0 <line_file> <plaintext_file> <output_cipher_bin>"
    exit 1
}
[[ $# -eq 3 ]] || usage

LINE_FILE="$1"
PLAINTEXT_FILE="$2"
OUT_CIPHER_BIN="$3"

# ----------------------------------------------------------------------
# Locate this script's directory and the utils folder
# ----------------------------------------------------------------------
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
UTILS_DIR="${SCRIPT_DIR}/utils"

# ----------------------------------------------------------------------
# Load mapping tables (card, die, coin) into associative arrays
# ----------------------------------------------------------------------
declare -A CARD_MAP DIE_MAP COIN_MAP

# --------- Card map ----------
while read -r token _; do
    # Skip empty lines or comment lines
    [[ -z "$token" || "$token" == \#* ]] && continue
    # Normalise token: upper‑case, replace Unicode suits with letters
    norm=$(echo "$token" | tr '[:lower:]' '[:upper:]' \
                       | sed -e 's/♣/C/g' -e 's/♦/D/g' -e 's/♥/H/g' -e 's/♠/S/g')
    CARD_MAP["$norm"]=1   # value is irrelevant; presence in the array is what we need
done < "${UTILS_DIR}/card_map.txt"

# --------- Die map ----------
while read -r token value; do
    [[ -z "$token" ]] && continue
    DIE_MAP["$token"]=$value
done < "${UTILS_DIR}/die_map.txt"

# --------- Coin map ----------
while read -r token value; do
    [[ -z "$token" ]] && continue
    COIN_MAP["$token"]=$value
done < "${UTILS_DIR}/coin_map.txt"

# ----------------------------------------------------------------------
# Convert the raw symbols line into a binary string
# ----------------------------------------------------------------------
bits=""

# Read the line, split on any whitespace, process token‑by‑token
lineno=0
while read -r raw_token; do
    ((lineno++))
    # Normalise the token the same way we did for the map
    token=$(echo "$raw_token" | tr '[:lower:]' '[:upper:]' \
                               | sed -e 's/♣/C/g' -e 's/♦/D/g' -e 's/♥/H/g' -e 's/♠/S/g')

    if [[ -n "${CARD_MAP[$token]+x}" ]]; then
        # Card → need the numeric value (0‑51).  Look it up in the original file.
        # We'll extract it with grep; this is cheap because the file is tiny.
        num=$(grep -E "^[[:space:]]*$raw_token[[:space:]]" "${UTILS_DIR}/card_map.txt" \
               | awk '{print $1}' | head -n1)
        # If the token was a Unicode glyph, the numeric value is the first field.
        # For aliases (e.g., KC) the numeric value is also the first field.
        # Guard against empty result (should never happen).
        if [[ -z "$num" ]]; then
            echo "ERROR: Could not find numeric value for card token \"$raw_token\" (line $lineno)." >&2
            exit 1
        fi
        bits+=$(printf "%06d" "$(bc <<< "obase=2;$((num))")")
    elif [[ -n "${DIE_MAP[$token]+x}" ]]; then
        # Die → 3‑bit
        num=${DIE_MAP[$token]}
        bits+=$(printf "%03d" "$(bc <<< "obase=2;$((num))")")
    elif [[ -n "${COIN_MAP[$token]+x}" ]]; then
        # Coin → single bit (already 0/1)
        bits+="${COIN_MAP[$token]}"
    else
        echo "ERROR: Unknown token \"$raw_token\" (line $lineno) in $LINE_FILE." >&2
        exit 1
    fi
done < <(tr -s '[:space:]' '\n' < "$LINE_FILE")

# ----------------------------------------------------------------------
# Pad the bit string to a whole number of bytes (8‑bit groups)
# ----------------------------------------------------------------------
rem=$(( ${#bits} % 8 ))
if (( rem != 0 )); then
    pad=$((8 - rem))
    bits+=$(printf "%0${pad}d" 0)
fi

# ----------------------------------------------------------------------
# Write the bit string to a temporary binary keystream file
# ----------------------------------------------------------------------
keystream_tmp=$(mktemp)
printf "%s" "$bits" | xxd -r -b > "$keystream_tmp"

# ----------------------------------------------------------------------
# Load plaintext, verify keystream length, truncate if necessary
# ----------------------------------------------------------------------
plain_len=$(stat -c%s "$PLAINTEXT_FILE")
key_len=$(stat -c%s "$keystream_tmp")

if (( key_len < plain_len )); then
    echo "ERROR: Keystream ($key_len bytes) shorter than plaintext ($plain_len bytes)." >&2
    echo "Add more symbols to the line or choose a longer line." >&2
    rm -f "$keystream_tmp"
    exit 2
fi

# Truncate keystream to exactly the plaintext length
keystream_trunc=$(mktemp)
dd if="$keystream_tmp" of="$keystream_trunc" bs=1 count="$plain_len" status=none
rm -f "$keystream_tmp"

# ----------------------------------------------------------------------
# XOR plaintext with keystream → ciphertext
# ----------------------------------------------------------------------
cipher_tmp=$(mktemp)

perl -e '
    open(my $pt, "<", $ARGV[0]) or die "cannot open plaintext: $!";
    open(my $ks, "<", $ARGV[1]) or die "cannot open keystream: $!";
    open(my $out, ">", $ARGV[2]) or die "cannot write ciphertext: $!";
    while (read($pt, my $pb, 1) && read($ks, my $kb, 1)) {
        print $out chr(ord($pb) ^ ord($kb));
    }
' "$PLAINTEXT_FILE" "$keystream_trunc" "$cipher_tmp"

mv "$cipher_tmp" "$OUT_CIPHER_BIN"
rm -f "$keystream_trunc"

# ----------------------------------------------------------------------
# Summary output
# ----------------------------------------------------------------------
echo "[INFO] Plaintext length : $plain_len bytes"
echo "[INFO] Keystream length : $key_len bytes (truncated to $plain_len for XOR)"
echo "[INFO] Ciphertext written to $OUT_CIPHER_BIN"
exit 0